import torch
from .custom_functions import \
    RayAABBIntersector, RayMarcher, VolumeRenderer
from einops import rearrange
import vren
import random
import tqdm
import math

MAX_SAMPLES = 1024
NEAR_DISTANCE = 0.01


def render(model, rays_o, rays_d, **kwargs):
    """
    Render rays by
    1. Compute the intersection of the rays with the scene bounding box
    2. Follow the process in @render_func (different for train/test)

    Inputs:
        model: NGP
        rays_o: (N_rays, 3) ray origins
        rays_d: (N_rays, 3) ray directions

    Outputs:
        result: dictionary containing final rgb and depth
    """
    rays_o = rays_o.contiguous(); rays_d = rays_d.contiguous()
    rays_d = torch.nn.functional.normalize(rays_d, dim=-1, eps=1e-15).contiguous()
    _, hits_t, _ = \
        RayAABBIntersector.apply(rays_o, rays_d, model.center, model.half_size, 1)
    hits_t[(hits_t[:, 0, 0]>=0)&(hits_t[:, 0, 0]<NEAR_DISTANCE), 0, 0] = NEAR_DISTANCE

    if kwargs.get('test_time', False):
        render_func = __render_rays_test
    else:
        render_func = __render_rays_train

    results = render_func(model, rays_o, rays_d, hits_t, **kwargs)
    if kwargs.get('test_time', False) and kwargs.get('density_scale', 1.0) != 1.0:
        _, hits_t, _ = \
            RayAABBIntersector.apply(rays_o, rays_d, model.center, model.half_size, 1)
        hits_t[(hits_t[:, 0, 0]>=0)&(hits_t[:, 0, 0]<NEAR_DISTANCE), 0, 0] = NEAR_DISTANCE
        _results = render_func(model, rays_o, rays_d, hits_t, **kwargs)
        results['depth'] = _results['depth']
        results['surface'] = _results['surface']
        
    for k, v in results.items():
        if kwargs.get('to_cpu', False):
            v = v.cpu()
            if kwargs.get('to_numpy', False):
                v = v.numpy()
        results[k] = v
    return results


@torch.no_grad()
def __render_rays_test(model, rays_o, rays_d, hits_t, **kwargs):
    """
    Render rays by

    while (a ray hasn't converged)
        1. Move each ray to its next occupied @N_samples (initially 1) samples 
           and evaluate the properties (sigmas, rgbs) there
        2. Composite the result to output; if a ray has transmittance lower
           than a threshold, mark this ray as converged and stop marching it.
           When more rays are dead, we can increase the number of samples
           of each marching (the variable @N_samples)
    """
    exp_step_factor = kwargs.get('exp_step_factor', 0.)
    black_bg = kwargs.get('black_bg', False)
    density_scale = kwargs.get('density_scale', 1.0)
    results = {}

    # output tensors to be filled in
    N_rays = len(rays_o)
    device = rays_o.device
    opacity = torch.zeros(N_rays, device=device)
    depth = torch.zeros(N_rays, device=device)
    rgb = torch.zeros(N_rays, 3, device=device)
    feature = torch.zeros(N_rays, 256, device=device)
    use_feature = model.feature_out_dim is not None

    samples = total_samples = 0
    alive_indices = torch.arange(N_rays, device=device)
    # if it's synthetic data, bg is majority so min_samples=1 effectively covers the bg
    # otherwise, 4 is more efficient empirically
    min_samples = 1 if exp_step_factor==0 else 4

    while samples < kwargs.get('max_samples', MAX_SAMPLES):
        N_alive = len(alive_indices)
        if N_alive==0: break

        # the number of samples to add on each ray
        N_samples = max(min(N_rays//N_alive, 64), min_samples)
        samples += N_samples

        xyzs, dirs, deltas, ts, N_eff_samples = \
            vren.raymarching_test(rays_o, rays_d, hits_t[:, 0], alive_indices,
                                  model.density_bitfield, model.cascades,
                                  model.scale, exp_step_factor,
                                  model.grid_size, MAX_SAMPLES, N_samples)
        total_samples += N_eff_samples.sum()
        xyzs = rearrange(xyzs, 'n1 n2 c -> (n1 n2) c')
        dirs = rearrange(dirs, 'n1 n2 c -> (n1 n2) c')
        valid_mask = ~torch.all(dirs==0, dim=1)
        if valid_mask.sum()==0: break

        sigmas = torch.zeros(len(xyzs), device=device)
        features = torch.zeros(len(xyzs), 256, device=device)
        rgbs = torch.zeros(len(xyzs), 3, device=device)
        _sigmas, _rgbs, _features = model(xyzs[valid_mask], dirs[valid_mask], **kwargs)

        sigmas[valid_mask], rgbs[valid_mask] = _sigmas.float(), _rgbs.float()
        sigmas = rearrange(sigmas, '(n1 n2) -> n1 n2', n2=N_samples)
        rgbs = rearrange(rgbs, '(n1 n2) c -> n1 n2 c', n2=N_samples)

        features[valid_mask] = _features.float()
        features = rearrange(features, '(n1 n2) c -> n1 n2 c', n2=N_samples)
        vren.composite_test_fw(
            sigmas.clone(), features.clone(), deltas.clone(), ts.clone(),
            hits_t[:, 0].clone(), alive_indices.clone(), kwargs.get('T_threshold', 1e-4),
            N_eff_samples.clone(), opacity.clone(), depth.clone(), feature)
            
        vren.composite_test_fw(
            sigmas * density_scale, rgbs, deltas, ts,
            hits_t[:, 0], alive_indices, kwargs.get('T_threshold', 1e-4),
            N_eff_samples, opacity, depth, rgb)
        alive_indices = alive_indices[alive_indices>=0] # remove converged rays

    results['opacity'] = opacity
    results['depth'] = depth
    results['rgb'] = rgb
    results['total_samples'] = total_samples # total samples for all rays
    results['rays_o'] = rays_o
    results['rays_d'] = rays_d
    
    feature_bg = torch.ones_like(feature) / math.sqrt(feature.shape[-1])
    feature += feature_bg*rearrange(1-opacity, 'n -> n 1')
    
    results['feature'] = feature / torch.linalg.vector_norm(feature, dim=1, keepdim=True)
    
    
    if kwargs.get('render_feature', False):
        with torch.no_grad():
            surfaces = rays_o + rays_d * results['depth'].detach()[..., None]
            results['surface'] = surfaces

    if exp_step_factor==0 and not black_bg: # synthetic
        rgb_bg = torch.ones(3, device=device)
    else: # real
        rgb_bg = torch.zeros(3, device=device)
    results['rgb'] += rgb_bg*rearrange(1-opacity, 'n -> n 1')

    return results


@torch.cuda.amp.autocast()
def __render_rays_train(model, rays_o, rays_d, hits_t, **kwargs):
    """
    Render rays by
    1. March the rays along their directions, querying @density_bitfield
       to skip empty space, and get the effective sample points (where
       there is object)
    2. Infer the NN at these positions and view directions to get properties
       (currently sigmas and rgbs)
    3. Use volume rendering to combine the result (front to back compositing
       and early stop the ray if its transmittance is below a threshold)
    """
    exp_step_factor = kwargs.get('exp_step_factor', 0.)
    results = {}
    black_bg = kwargs.get('black_bg', False)

    rays_a, xyzs, dirs, results['deltas'], results['ts'], results['rm_samples'] = \
        RayMarcher.apply(
            rays_o, rays_d, hits_t[:, 0], model.density_bitfield,
            model.cascades, model.scale,
            exp_step_factor, model.grid_size, MAX_SAMPLES)

    for k, v in kwargs.items(): # supply additional inputs, repeated per ray
        if isinstance(v, torch.Tensor) and 'seg' not in k:
            kwargs[k] = torch.repeat_interleave(v[rays_a[:, 0]], rays_a[:, 2], 0)
    
    sigmas, rgbs, features = model(xyzs, dirs, **kwargs)

    (results['vr_samples'], # volume rendering effective samples
    results['opacity'], results['depth'], results['rgb'], results['ws']) = \
        VolumeRenderer.apply(sigmas.contiguous(), rgbs.contiguous(), results['deltas'], results['ts'],
                             rays_a, kwargs.get('T_threshold', 1e-4))
    results['rays_a'] = rays_a
    results['rays_d'] = rays_d

    _, _, _, feature, _ = VolumeRenderer.apply(sigmas.contiguous().detach(), features.contiguous(), results['deltas'].detach(), results['ts'].detach(),
                             rays_a.detach(), kwargs.get('T_threshold', 1e-4))
    feature_bg = torch.ones_like(feature) / math.sqrt(feature.shape[-1])
    feature = feature + feature_bg*rearrange(1-results['opacity'].detach(), 'n -> n 1')
    results['norm'] = torch.linalg.vector_norm(feature, dim=1, keepdim=True)
    results['feature'] = feature / (torch.linalg.vector_norm(feature, dim=1, keepdim=True) + 1e-6)
    if torch.isnan(results['feature']).any():
        import pdb
        pdb.set_trace()
            
    if exp_step_factor==0: # synthetic
        if black_bg:
            rgb_bg = torch.zeros(3, device=rays_o.device)
        else:
            rgb_bg = torch.ones(3, device=rays_o.device)
    else: # real
        if kwargs.get('random_bg', False):
            rgb_bg = torch.rand(3, device=rays_o.device)
        else:
            rgb_bg = torch.zeros(3, device=rays_o.device)
    results['rgb'] = results['rgb'] + \
                     rgb_bg*rearrange(1-results['opacity'], 'n -> n 1')

    if 'depth_smooth_samples_num' in kwargs:
        results['rgb'] = results['rgb'][:-kwargs['depth_smooth_samples_num']]
        results['opacity_sup'] = results['opacity'][-kwargs['depth_smooth_samples_num']:]
        results['opacity'] = results['opacity'][:-kwargs['depth_smooth_samples_num']]
        results['ws'] = results['ws'][:-kwargs['depth_smooth_samples_num']]
        results['ts'] = results['ts'][:-kwargs['depth_smooth_samples_num']]
        results['deltas'] = results['deltas'][:-kwargs['depth_smooth_samples_num']]
        results['rays_a'] = results['rays_a'][:-kwargs['depth_smooth_samples_num']]
        results['rays_d'] = results['rays_d'][:-kwargs['depth_smooth_samples_num']]
        results['feature'] = results['feature'][:-kwargs['depth_smooth_samples_num']]
        results['depth_sup'] = results['depth'][-kwargs['depth_smooth_samples_num']:]
        
    return results
